// ShortCodeIframe 插件 JavaScript 代码

// 页面加载完成后执行
document.addEventListener('DOMContentLoaded', function() {
    // 初始化视频切换功能
    initVideoTabs();
    // 初始化HLS播放器
    initHlsPlayers();
});

/**
 * 初始化视频切换功能
 */
function initVideoTabs() {
    // 获取所有视频容器，包括video-container和hls-container
    var videoContainers = document.querySelectorAll('.video-container, .hls-container');
    
    // 使用传统的for循环替代forEach，确保兼容性
    for (var i = 0; i < videoContainers.length; i++) {
        var container = videoContainers[i];
        // 为每个视频容器初始化切换功能
        initVideoContainer(container);
    }
}

/**
 * 初始化所有HLS播放器
 */
function initHlsPlayers() {
    // 获取所有HLS播放器容器
    var hlsContainers = document.querySelectorAll('.hls-container');
    
    // 使用传统的for循环，确保兼容性
    for (var i = 0; i < hlsContainers.length; i++) {
        var container = hlsContainers[i];
        // 初始化ArtPlayer播放器
        initializeArtPlayer(container);
    }
}

/**
 * 解码URL编码的字符串
 * @param {string} str 编码的字符串
 * @returns {string} 解码后的字符串
 */
function decodeURIComponent(str) {
    // 创建一个临时textarea元素来解码HTML实体
    var textarea = document.createElement('textarea');
    textarea.innerHTML = str;
    return textarea.value;
}

/**
 * 初始化单个视频容器
 * @param {HTMLElement} container 视频容器元素
 */
function initVideoContainer(container) {
    // 检查是否已经初始化过，避免重复初始化
    if (container.dataset.videoInitialized === 'true') {
        return;
    }
    
    // 标记为已初始化
    container.dataset.videoInitialized = 'true';
    
    // 获取视频切换标签
    var tabs = container.querySelectorAll('.video-tab');
    
    // 如果没有标签，不需要初始化
    if (tabs.length === 0) {
        return;
    }
    
    // 为每个标签绑定点击事件
    for (var i = 0; i < tabs.length; i++) {
        (function(index) {
            var tab = tabs[index];
            tab.addEventListener('click', function(e) {
                e.preventDefault();
                // 切换视频
                switchVideo(container, index);
            });
        })(i);
    }
}

/**
 * 切换视频
 * @param {HTMLElement} container 视频容器元素
 * @param {number} index 视频索引
 */
function switchVideo(container, index) {
    // 获取视频URL列表
    var urlsInput = container.querySelector('.video-urls');
    var titlesInput = container.querySelector('.video-titles');
    
    if (!urlsInput || !titlesInput) {
        return;
    }
    
    // 解码URL列表（处理可能存在的编码）
    var urlsStr = decodeURIComponent(urlsInput.value);
    var urls = urlsStr.split(',');
    
    // 解码标题列表
    var titlesStr = decodeURIComponent(titlesInput.value);
    var titles = titlesStr.split('|');
    
    // 获取所有标签
    var tabs = container.querySelectorAll('.video-tab');
    
    // 验证索引是否有效
    if (index < 0 || index >= urls.length) {
        return;
    }
    
    // 更新标签激活状态
    for (var i = 0; i < tabs.length; i++) {
        if (i === index) {
            tabs[i].classList.add('active');
        } else {
            tabs[i].classList.remove('active');
        }
    }
    
    // 获取当前选中的URL
    var currentUrl = urls[index].trim();
    
    // 获取解析器URL（已在PHP端根据视频类型设置，直接使用即可）
    var parserUrlInput = container.querySelector('.video-parser-url');
    var parserUrl = parserUrlInput ? parserUrlInput.value : (window.shortCodeIframeParserUrl || '');
    
    // 检查是否使用解析地址
    var useParserInput = container.querySelector('.video-use-parser');
    var useParser = useParserInput ? (useParserInput.value === 'true') : true;
    
    // 判断容器类型
    if (container.classList.contains('hls-container')) {
        // HLS容器，使用ArtPlayer
        switchHlsVideo(container, index, urls, parserUrl, useParser);
    } else if (container.classList.contains('video-container')) {
        // 普通视频容器，使用iframe
        switchIframeVideo(container, index, currentUrl, urls, parserUrl, useParser);
    }
}

/**
 * 切换HLS视频（ArtPlayer）
 * @param {HTMLElement} container 视频容器元素
 * @param {number} index 视频索引
 * @param {Array} urls 视频URL列表
 * @param {string} parserUrl 解析器URL
 * @param {boolean} useParser 是否使用解析器
 */
function switchHlsVideo(container, index, urls, parserUrl, useParser) {
    var artPlayer = container.artPlayer;
    var videoUrl = urls[index];
    
    if (!artPlayer || !videoUrl) {
        return;
    }
    
    // 获取解析后的视频URL
    var finalUrl = getParsedVideoUrl(videoUrl, parserUrl, useParser);
    var videoType = getVideoType(finalUrl);
    
    // 切换视频
    artPlayer.switchUrl(finalUrl, videoType);
    artPlayer.play();
    
    // 加载新视频的弹幕数据
    var originalUrl = urls[index];
    var danmuApiUrl = `https://dm.ruyijx.com/?ac=dm&url=${encodeURIComponent(originalUrl)}`;
    
    fetch(danmuApiUrl)
        .then(response => response.json())
        .then(danmuData => {
            // 转换弹幕格式
            const modeMap = {
                'top': 1,
                'bottom': 2,
                'scroll': 0,
                'right': 0
            };
            
            const convertedDanmu = danmuData.danmuku.map(danmuItem => ({
                time: danmuItem[0],
                mode: modeMap[danmuItem[1]] || 0,
                color: danmuItem[2],
                text: danmuItem[4]
            }));
            
            // 按照官方文档方式切换新弹幕
            if (artPlayer.plugins.artplayerPluginDanmuku) {
                artPlayer.plugins.artplayerPluginDanmuku.config({
                    danmuku: convertedDanmu,
                });
                artPlayer.plugins.artplayerPluginDanmuku.load();
            }
        })
        .catch(error => {
            console.error('Failed to load danmu data for video:', error);
            // 加载失败时，清空弹幕
            if (artPlayer.plugins.artplayerPluginDanmuku) {
                artPlayer.plugins.artplayerPluginDanmuku.config({
                    danmuku: [],
                });
                artPlayer.plugins.artplayerPluginDanmuku.load();
            }
        });
    
    // 更新当前视频索引
    container.currentVideoIndex = index;
}

/**
 * 切换普通视频（iframe）
 * @param {HTMLElement} container 视频容器元素
 * @param {number} index 视频索引
 * @param {string} currentUrl 当前视频URL
 * @param {Array} urls 视频URL列表
 * @param {string} parserUrl 解析器URL
 * @param {boolean} useParser 是否使用解析器
 */
function switchIframeVideo(container, index, currentUrl, urls, parserUrl, useParser) {
    // 获取视频播放器
    var iframe = container.querySelector('.iframe_video');
    if (!iframe) {
        return;
    }
    
    // 获取loading动画元素
    var loading = container.querySelector('.video-loading');
    
    // 显示loading动画
    if (loading) {
        loading.classList.add('show');
    }
    
    // 移除之前的事件监听器，防止多次触发
    iframe.onload = null;
    iframe.onerror = null;
    
    // 更彻底的资源释放：
    // 1. 先清空iframe的src，触发资源释放
    iframe.src = 'about:blank';
    
    // 2. 强制刷新iframe内容，确保资源被释放
    if (iframe.contentWindow) {
        try {
            iframe.contentWindow.document.write('');
            iframe.contentWindow.document.close();
        } catch (e) {
            // 忽略跨域错误
        }
    }
    
    // 3. 等待一小段时间，确保资源释放完成
    setTimeout(function() {
        // 4. 显示loading动画
        if (loading) {
            loading.classList.add('show');
        }
        
        // 5. 最后设置新的URL
        // 如果使用解析地址且解析地址不为空，使用解析地址+原始URL，否则使用原始URL
        var finalUrl = (useParser && parserUrl) ? (parserUrl + encodeURIComponent(currentUrl)) : currentUrl;
        iframe.src = finalUrl;
    }, 100);
    
    // 监听视频加载完成事件，隐藏loading动画
    iframe.onload = function() {
        if (loading) {
            loading.classList.remove('show');
        }
    };
    
    // 设置加载错误处理
    iframe.onerror = function() {
        if (loading) {
            loading.classList.remove('show');
        }
        // 可以在这里添加错误提示
        console.error('视频加载失败: ' + currentUrl);
    };
    
    // 设置超时，防止视频加载失败导致loading动画一直显示
    setTimeout(function() {
        if (loading && loading.classList.contains('show')) {
            loading.classList.remove('show');
        }
    }, 10000); // 10秒后自动隐藏
}

/**
 * 初始化ArtPlayer播放器
 * @param {HTMLElement} container - HLS播放器容器
 */
function initializeArtPlayer(container) {
    var id = container.id;
    var artPlayerId = 'artplayer-' + id;
    var artPlayerContainer = document.getElementById(artPlayerId);
    
    if (!artPlayerContainer) {
        console.error('ArtPlayer容器未找到: ' + artPlayerId);
        return;
    }
    
    // 获取视频数据
    var videoUrlsInput = container.querySelector('.video-urls');
    var videoTitlesInput = container.querySelector('.video-titles');
    var parserUrlInput = container.querySelector('.video-parser-url');
    var useParserInput = container.querySelector('.video-use-parser');
    
    if (!videoUrlsInput) {
        console.error('视频URL数据未找到: ' + id);
        return;
    }
    
    var videoUrlsStr = decodeURIComponent(videoUrlsInput.value);
    var videoUrls = videoUrlsStr.split(',').filter(function(url) { return url.trim() !== ''; });
    
    var videoTitlesStr = videoTitlesInput ? decodeURIComponent(videoTitlesInput.value) : '';
    var videoTitles = videoTitlesStr.split('|');
    
    // 解析地址已在PHP端根据视频类型设置，直接使用即可
    var parserUrl = parserUrlInput ? parserUrlInput.value : (window.shortCodeIframeParserUrl || '');
    var useParser = useParserInput ? (useParserInput.value === 'true') : true;
    
    if (videoUrls.length === 0) {
        console.error('视频URL列表为空: ' + id);
        return;
    }
    
    // 获取第一个视频URL
    var firstVideoUrl = getParsedVideoUrl(videoUrls[0], parserUrl, useParser);
    var videoType = getVideoType(firstVideoUrl);
    
    // 加载并转换弹幕数据
    var originalUrl = videoUrls[0];
    var danmuApiUrl = `https://dm.ruyijx.com/?ac=dm&url=${encodeURIComponent(originalUrl)}`;
    
    // 检测是否为移动端设备
    var isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
    
    // 初始化函数，支持有弹幕和无弹幕两种情况
    function initPlayerWithDanmu(danmuData) {
        // 初始化ArtPlayer
        var art = new Artplayer({
            container: '#' + artPlayerId, // 播放器容器元素
            url: firstVideoUrl, // 视频播放地址
            type: 'm3u8', // 视频类型（使用自动检测的类型）
            autoplay: true, // 自动播放
            autoSize: false, // 禁用自动大小调整，强制铺满容器
            playbackRate: true, // 显示播放速度控制
            setting: true, // 显示设置菜单
            pip: !isMobile, // 画中画：移动端不显示
            aspectRatio: true, // 允许调整视频宽高比
            fullscreen: true, // 启用视频全屏功能
            fullscreenWeb: !isMobile, // 网页全屏：移动端不显示
            playsInline: true, // 允许网页内播放（移动端）
            autoPlayback: true, // 自动回放（记忆播放）
            theme: '#23ade5', // 播放器主题颜色
            lang: navigator.language.toLowerCase(), // 根据浏览器自动设置语言
            customType: {
                m3u8: function (video, url) {
                    if (window.Hls && window.Hls.isSupported()) {
                        var hls = new Hls({
                            maxBufferLength: 300,       // 正常缓冲时长 (秒)
                            maxMaxBufferLength: 600,    // 最大缓冲时长 (秒)
                            fragLoadingSetup: function(xhr, frag) {
                                // 为伪装成PNG的TS片段添加正确的请求头
                                if (frag.url && frag.url.toLowerCase().endsWith('.png')) {
                                    xhr.setRequestHeader('Accept', '*/*');
                                    xhr.setRequestHeader('Content-Type', 'video/mp2t');
                                }
                            }
                        });
                        hls.loadSource(url);
                        hls.attachMedia(video);
                    } else if (video.canPlayType('application/vnd.apple.mpegurl')) {
                        video.src = url;
                    }
                },
                flv: function (video, url) {
                    if (window.flvjs && window.flvjs.isSupported()) {
                        var flvPlayer = flvjs.createPlayer({
                            type: 'flv',
                            url: url
                        });
                        flvPlayer.attachMediaElement(video);
                        flvPlayer.load();
                    }
                }
            },
            plugins: [
                artplayerPluginDanmuku({
                    danmuku: danmuData,
                    
                    // 以下为非必填
                    speed: 8, // 弹幕持续时间，范围在[1 ~ 10]，数值越大速度越慢
                    margin: [10, '25%'], // 弹幕上下边距，支持像素数字和百分比
                    opacity: 1, // 弹幕透明度，范围在[0 ~ 1]，80%不透明度
                    color: '#FFFFFF', // 默认弹幕颜色，可以被单独弹幕项覆盖
                    mode: 1, // 默认弹幕模式: 0: 滚动，1: 顶部，2: 底部
                    modes: [0, 1, 2], // 弹幕可见的模式
                    fontSize: '3%', // 弹幕字体大小，支持像素数字和百分比，根据窗口大小自动缩放
                    antiOverlap: true, // 弹幕是否防重叠
                    synchronousPlayback: true, // 是否同步播放速度
                    mount: undefined, // 弹幕发射器挂载点, 默认为播放器控制栏中部
                    heatmap: false, // 是否开启热力图
                    width: 512, // 当播放器宽度小于此值时，弹幕发射器置于播放器底部
                    points: [], // 热力图数据
                    filter: danmu => danmu.text.length <= 100, // 弹幕载入前的过滤器
                    beforeVisible: () => true, // 弹幕显示前的过滤器，返回 true 则可以发送
                    visible: true, // 弹幕层是否可见
                    emitter: true, // 是否开启弹幕发射器
                    maxLength: 200, // 弹幕输入框最大长度, 范围在[1 ~ 1000]
                    lockTime: 5, // 输入框锁定时间，范围在[1 ~ 60]
                    theme: 'dark', // 弹幕主题，支持 dark 和 light，只在自定义挂载时生效
                    OPACITY: {}, // 不透明度配置项
                    MARGIN: {}, // 显示区域配置项
                    SPEED: {}, // 弹幕速度配置项
                    COLOR: [], // 颜色列表配置项
                    
                    // 手动发送弹幕前的过滤器，返回 true 则可以发送，可以做存库处理
                    beforeEmit(danmu) {
                        return new Promise((resolve) => {
                            console.log(danmu)
                            setTimeout(() => {
                                resolve(true)
                            }, 1000)
                        })
                    },
                }),
            ],
        });
        
        // 将播放器实例存储在容器上，以便后续切换视频使用
        container.artPlayer = art;
        container.videoUrls = videoUrls;
        container.videoTitles = videoTitles;
        container.parserUrl = parserUrl;
        container.useParser = useParser;
        container.currentVideoIndex = 0;
    }
    
    // 尝试从API获取弹幕数据
    fetch(danmuApiUrl)
        .then(response => response.json())
        .then(danmuData => {
            // 转换弹幕格式
            const modeMap = {
                'top': 1,
                'bottom': 2,
                'scroll': 0,
                'right': 0
            };
            const convertedDanmu = danmuData.danmuku.map(danmuItem => ({
                time: danmuItem[0],
                mode: modeMap[danmuItem[1]] || 0,
                color: danmuItem[2],
                text: danmuItem[4]
            }));
            
            // 使用转换后的弹幕数据初始化播放器
            initPlayerWithDanmu(convertedDanmu);
        })
        .catch(error => {
            console.error('Failed to load danmu data:', error);
            // 加载失败时，使用空弹幕数据初始化播放器
            initPlayerWithDanmu([]);
        });
}

/**
 * 获取解析后的视频URL
 * @param {string} originalUrl - 原始视频URL
 * @param {string} parserUrl - 解析地址前缀
 * @param {boolean} useParser - 是否使用解析地址
 * @returns {string} 解析后的视频URL
 */
function getParsedVideoUrl(originalUrl, parserUrl, useParser) {
    if (useParser && parserUrl) {
        // 使用解析地址
        return parserUrl + encodeURIComponent(originalUrl);
    }
    // 直接使用原始地址
    return originalUrl;
}

/**
 * 获取视频类型
 * @param {string} url - 视频URL
 * @returns {string} 视频类型（m3u8、flv或auto）
 */
function getVideoType(url) {
    if (url.toLowerCase().endsWith('.m3u8')) {
        return 'm3u8';
    } else if (url.toLowerCase().endsWith('.flv')) {
        return 'flv';
    } else {
        return 'auto';
    }
}

/**
 * 扩展ArtPlayer，添加切换URL方法
 * 如果ArtPlayer实例没有switchUrl方法，则添加该方法
 */
if (typeof Artplayer !== 'undefined') {
    // 检查switchUrl方法是否存在，不存在则添加
    if (typeof Artplayer.prototype.switchUrl === 'undefined') {
        Artplayer.prototype.switchUrl = function(url, type) {
            this.url = url;
            this.type = type;
            
            // 重新加载视频
            this.load();
            
            return this;
        };
    }
}