<?php
/**
 * 短代码Iframe插入视频到文章中
 *
 * @package 短代码Iframe插入视频
 * @author xiao
 * @version 1.0.1
 * @link http://wodecuo.chez.com
 */

if (!defined('__TYPECHO_ROOT_DIR__')) exit;

/**
 * 短代码插入视频 插件实现
 *
 * @package 短代码Iframe插入视频
 */
class ShortCodeIframe_Plugin implements Typecho_Plugin_Interface
{
    /**
     * 激活插件方法,如果激活失败,直接抛出异常
     *
     * @access public
     * @return string
     * @throws Typecho_Plugin_Exception
     */
    public static function activate()
    {
        // 注册内容过滤钩子，使用content而不是contentEx
        Typecho_Plugin::factory('Widget_Abstract_Contents')->content = array('ShortCodeIframe_Plugin', 'parseContent');
        Typecho_Plugin::factory('Widget_Abstract_Contents')->excerpt = array('ShortCodeIframe_Plugin', 'parseExcerpt');
        
        // 注册header钩子，用于输出CSS样式
        Typecho_Plugin::factory('Widget_Archive')->header = array('ShortCodeIframe_Plugin', 'outputHeader');
        
        return _t('ShortCodeIframe 插件已激活，现在可以使用 [video]URL[/video] 和 [hls]URL[/hls] 短代码嵌入视频。已支持两种独立的解析地址配置。');
    }

    /**
     * 禁用插件方法,如果禁用失败,直接抛出异常
     *
     * @static
     * @access public
     * @return string
     * @throws Typecho_Plugin_Exception
     */
    public static function deactivate()
    {
        return _t('ShortCodeIframe 插件已禁用。');
    }

    /**
     * 获取插件配置面板
     *
     * @access public
     * @param Typecho_Widget_Helper_Form $form 配置面板
     * @return void
     */
    public static function config(Typecho_Widget_Helper_Form $form)
    {
        // video视频解析地址设置
        $videoParserUrl = new Typecho_Widget_Helper_Form_Element_Text('videoParserUrl', NULL, 'https://jx.hls.one/?url=', _t('Video视频解析地址'), _t('请输入Video视频解析地址前缀，例如：https://jx.hls.one/?url='));
        $form->addInput($videoParserUrl);
        
        // hls视频解析地址设置
        $hlsParserUrl = new Typecho_Widget_Helper_Form_Element_Text('hlsParserUrl', NULL, '', _t('HLS视频解析地址'), _t('请输入HLS视频解析地址前缀，例如：v.php?url='));
        $form->addInput($hlsParserUrl);
        
        // 插件使用说明 - 直接显示在页面上
        echo '<div style="margin: 15px 0; padding: 10px; background: #f8f9fa; border: 1px solid #eee; border-radius: 4px;">';
        echo '<h3 style="margin-top: 0; color: #333;">ShortCodeIframe 插件使用说明</h3>';
        echo '<div style="color: #666;">';
        
        // 配置说明
        echo '<h4>配置说明</h4>';
        echo '<p>插件现在支持两种独立的解析地址配置：</p>';
        echo '<ul style="margin: 10px 0; padding-left: 20px;">';
        echo '<li><strong>Video视频解析地址</strong> - 用于[video]短代码的解析地址</li>';
        echo '<li><strong>HLS视频解析地址</strong> - 用于[hls]短代码的解析地址</li>';
        echo '</ul>';
        
        // 支持的短代码类型
        echo '<h4>支持的短代码类型</h4>';
        echo '<ul style="margin: 10px 0; padding-left: 20px;">';
        echo '<li><strong>[video]URL[/video]</strong> - 使用iframe播放普通视频</li>';
        echo '<li><strong>[hls]URL[/hls]</strong> - 使用ArtPlayer播放m3u8/flv视频</li>';
        echo '</ul>';
        
        // 单个视频 - video
        echo '<h4>1. 单个视频（iframe播放）</h4>';
        echo '<pre style="background: #fff; padding: 10px; border: 1px solid #ddd; border-radius: 3px;">[video]https://example.com/video1.mp4[/video]</pre>';
        
        // 单个视频 - hls
        echo '<h4>2. 单个HLS视频（ArtPlayer播放）</h4>';
        echo '<pre style="background: #fff; padding: 10px; border: 1px solid #ddd; border-radius: 3px;">[hls]https://example.com/video1.m3u8[/hls]</pre>';
        
        // 不使用解析地址
        echo '<h4>3. 不使用解析地址</h4>';
        echo '<pre style="background: #fff; padding: 10px; border: 1px solid #ddd; border-radius: 3px;">[!video]https://example.com/video1.mp4[/!video]</pre>';
        echo '<pre style="background: #fff; padding: 10px; border: 1px solid #ddd; border-radius: 3px;">[!hls]https://example.com/video1.m3u8[/!hls]</pre>';
        echo '<p style="color: #666; margin: 5px 0 15px 0;">在video或hls前添加!符号，表示不使用配置的解析地址，直接使用原始链接</p>';
        
        // 带标题的单个视频
        echo '<h4>4. 带标题的单个视频</h4>';
        echo '<pre style="background: #fff; padding: 10px; border: 1px solid #ddd; border-radius: 3px;">[video name="我的视频"]https://example.com/video1.mp4[/video]</pre>';
        echo '<pre style="background: #fff; padding: 10px; border: 1px solid #ddd; border-radius: 3px;">[hls name="我的视频"]https://example.com/video1.m3u8[/hls]</pre>';
        
        // 多个视频（分集）
        echo '<h4>5. 多个视频（分集）</h4>';
        echo '<pre style="background: #fff; padding: 10px; border: 1px solid #ddd; border-radius: 3px;">[video name="我的剧集"]
https://example.com/episode1.mp4
https://example.com/episode2.mp4
https://example.com/episode3.mp4
[/video]</pre>';
        echo '<pre style="background: #fff; padding: 10px; border: 1px solid #ddd; border-radius: 3px;">[hls name="我的剧集"]
https://example.com/episode1.m3u8
https://example.com/episode2.m3u8
https://example.com/episode3.m3u8
[/hls]</pre>';
        
        // 自定义分集标题
        echo '<h4>6. 自定义分集标题</h4>';
        echo '<pre style="background: #fff; padding: 10px; border: 1px solid #ddd; border-radius: 3px;">[video name="我的剧集"]
第一集标题|https://example.com/episode1.mp4
第二集标题|https://example.com/episode2.mp4
第三集标题|https://example.com/episode3.mp4
[/video]</pre>';
        echo '<pre style="background: #fff; padding: 10px; border: 1px solid #ddd; border-radius: 3px;">[hls name="我的剧集"]
第一集标题|https://example.com/episode1.m3u8
第二集标题|https://example.com/episode2.m3u8
第三集标题|https://example.com/episode3.m3u8
[/hls]</pre>';
        
        // 注意事项
        echo '<h4>注意事项</h4>';
        echo '<ul style="margin: 10px 0; padding-left: 20px;">';
        echo '<li>视频URL必须是完整的、可访问的URL</li>';
        echo '<li>支持mp4、m3u8、flv等多种视频格式</li>';
        echo '<li>多个视频URL之间可以用换行或逗号分隔</li>';
        echo '<li>name属性用于显示视频标题，可选</li>';
        echo '<li>id属性用于标识视频容器，可选</li>';
        echo '<li>自定义分集标题格式：标题|URL</li>';
        echo '<li>在video或hls前添加!符号，表示不使用配置的解析地址，直接使用原始链接</li>';
        echo '<li>ArtPlayer支持播放速度控制、画中画、全屏等功能</li>';
        echo '</ul>';
        
        echo '</div>';
        echo '</div>';
        
        // 输出激活提示
        echo '<div style="margin: 15px 0; padding: 10px; background: #d4edda; border: 1px solid #c3e6cb; border-radius: 4px; color: #155724;">';
        echo '<strong>插件已更新！</strong> 现在支持两种短代码类型：[video]用于iframe播放，[hls]用于ArtPlayer播放。';
        echo '</div>';
    }

    /**
     * 个人用户的配置面板
     *
     * @access public
     * @param Typecho_Widget_Helper_Form $form
     * @return void
     */
    public static function personalConfig(Typecho_Widget_Helper_Form $form){}

    /**
     * 短代码处理函数
     *
     * @access private
     * @param string $content 原始内容
     * @return string 处理后的内容
     */
    private static function parseShortCode($content)
    {
        // 定义短代码正则表达式，支持多个视频和分集标识
        // 支持两种格式：[video]和[!video]，后者表示不使用解析地址
        $videoPattern = '/\[(!?video)\s*(?:id="([^"]*)")?\s*(?:name="([^"]*)")?\s*\](.*?)\[\/video\]/s';
        
        // 处理video短代码，支持多个视频
        $content = preg_replace_callback($videoPattern, array('ShortCodeIframe_Plugin', 'parseVideoShortCode'), $content);
        
        // 定义hls短代码正则表达式，支持多个视频和分集标识
        // 支持两种格式：[hls]和[!hls]，后者表示不使用解析地址
        $hlsPattern = '/\[(!?hls)\s*(?:id="([^"]*)")?\s*(?:name="([^"]*)")?\s*\](.*?)\[\/hls\]/s';
        
        // 处理hls短代码，支持多个视频
        $content = preg_replace_callback($hlsPattern, array('ShortCodeIframe_Plugin', 'parseHlsShortCode'), $content);
        
        return $content;
    }
    
    /**
     * 单个video短代码处理函数
     *
     * @access private
     * @param array $matches 匹配结果
     * @return string 处理后的HTML
     */
    private static function parseVideoShortCode($matches)
    {
        return self::parseMediaShortCode($matches, 'video');
    }
    
    /**
     * 单个hls短代码处理函数
     *
     * @access private
     * @param array $matches 匹配结果
     * @return string 处理后的HTML
     */
    private static function parseHlsShortCode($matches)
    {
        return self::parseMediaShortCode($matches, 'hls');
    }
    
    /**
     * 通用媒体短代码处理函数
     *
     * @access private
     * @param array $matches 匹配结果
     * @param string $type 媒体类型：video或hls
     * @return string 处理后的HTML
     */
    private static function parseMediaShortCode($matches, $type)
    {
        $shortCodeType = $matches[1]; // 可能是 "video"、"!video"、"hls" 或 "!hls"
        $id = $matches[2] ? $matches[2] : $type . '_' . uniqid();
        $name = $matches[3] ? $matches[3] : '';
        $content = trim($matches[4]);
        
        // 解析视频数据，支持自定义分集标题
        $videos = array();
        $lines = preg_split('/[\r\n]+/', $content);
        
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) {
                continue;
            }
            
            // 支持两种格式：
            // 1. URL（旧格式，兼容）
            // 2. 标题|URL（新格式，支持自定义标题）
            if (strpos($line, '|') !== false) {
                list($title, $url) = explode('|', $line, 2);
                $title = trim($title);
                $url = trim($url);
            } else {
                $title = '';
                $url = trim($line);
            }
            
            if (!empty($url)) {
                $videos[] = array('title' => $title, 'url' => $url);
            }
        }
        
        if (empty($videos)) {
            return '';
        }
        
        // 获取配置的解析地址
        $options = Typecho_Widget::widget('Widget_Options');
        
        // 根据视频类型获取对应的解析地址
        if ($type === 'hls') {
            $parserUrl = $options->plugin('ShortCodeIframe')->hlsParserUrl;
        } else {
            $parserUrl = $options->plugin('ShortCodeIframe')->videoParserUrl;
        }
        
        // 如果是 [!video] 或 [!hls] 格式，不使用解析地址
        $useParserUrl = substr($shortCodeType, 0, 1) !== '!';
        $actualType = substr($shortCodeType, 0, 1) === '!' ? substr($shortCodeType, 1) : $shortCodeType;
        
        // 生成视频容器HTML
        $html = '<div class="' . $type . '-container" id="' . $id . '" data-use-parser-url="' . ($useParserUrl ? 'true' : 'false') . '" data-type="' . $type . '">';
        if (!empty($name)) {
            $html .= '<div class="' . $type . '-title">' . $name . '</div>';
        }
        
        if ($type === 'hls') {
            // 生成artplayer播放器
            $html .= '<div class="hls-player-wrapper">';
            $html .= '<div id="artplayer-' . $id . '" class="artplayer"></div>';
            
            // 生成视频URL和标题数据
            $videoUrls = array();
            $videoTitles = array();
            foreach ($videos as $video) {
                $videoUrls[] = $video['url'];
                $videoTitles[] = $video['title'];
            }
            
            $html .= '<input type="hidden" class="video-urls" value="' . implode(',', $videoUrls) . '" />';
            $html .= '<input type="hidden" class="video-titles" value="' . implode('|', $videoTitles) . '" />';
            $html .= '<input type="hidden" class="video-parser-url" value="' . $parserUrl . '" />';
            $html .= '<input type="hidden" class="video-use-parser" value="' . ($useParserUrl ? 'true' : 'false') . '" />';
            $html .= '</div>';
        } else {
            // 生成iframe播放器
            $html .= '<div class="iframe_div iframe_div-16x9">';
            $html .= '<div class="video-loading"></div>';
            // 如果使用解析地址且解析地址不为空，使用解析地址+原始URL，否则使用原始URL
            $firstVideoUrl = ($useParserUrl && !empty($parserUrl)) ? $parserUrl . $videos[0]['url'] : $videos[0]['url'];
            $html .= '<iframe class="iframe_video" src="' . $firstVideoUrl . '" frameborder="0" allow="fullscreen; autoplay; encrypted-media"></iframe>';
            
            // 生成视频URL和标题数据
            $videoUrls = array();
            $videoTitles = array();
            foreach ($videos as $video) {
                $videoUrls[] = $video['url'];
                $videoTitles[] = $video['title'];
            }
            
            $html .= '<input type="hidden" class="video-urls" value="' . implode(',', $videoUrls) . '" />';
            $html .= '<input type="hidden" class="video-titles" value="' . implode('|', $videoTitles) . '" />';
            $html .= '<input type="hidden" class="video-parser-url" value="' . $parserUrl . '" />';
            $html .= '<input type="hidden" class="video-use-parser" value="' . ($useParserUrl ? 'true' : 'false') . '" />';
            $html .= '</div>';
        }
        
        // 生成视频切换菜单，放在播放器下面
        if (count($videos) > 1) {
            $html .= '<div class="video-tabs">';
            foreach ($videos as $index => $video) {
                $tabClass = $index == 0 ? 'active' : '';
                // 如果有自定义标题，使用自定义标题，否则使用默认标题
                $tabTitle = !empty($video['title']) ? $video['title'] : '第' . ($index + 1) . '集';
                $html .= '<span class="video-tab ' . $tabClass . '" data-index="' . $index . '" data-container="' . $id . '">' . $tabTitle . '</span>';
            }
            $html .= '</div>';
        }
        
        // 输出JavaScript配置
        $html .= '<script type="text/javascript">var shortCodeIframeParserUrl = "' . $parserUrl . '";</script>';
        
        $html .= '</div>';
        
        return $html;
    }

    /**
     * 处理文章内容
     *
     * @access public
     * @param string $content 文章内容
     * @param Widget_Abstract_Contents $widget 内容对象
     * @return string 处理后的内容
     */
    public static function parseContent($content, $widget)
    {
        // 先处理短代码
        $content = self::parseShortCode($content);
        
        // Typecho的content钩子机制：如果有插件处理了content钩子，Typecho会跳过默认的Markdown解析
        // 所以我们需要检查是否需要自己处理Markdown解析
        if (method_exists($widget, 'markdown') && method_exists($widget, 'autoP')) {
            // 检查是否有其他插件已经处理过
            $handlers = Typecho_Plugin::export();
            $contentHandlers = $handlers['handles']['Widget_Abstract_Contents:content'];
            
            // 查找当前插件在处理器列表中的位置
            $currentIndex = -1;
            foreach ($contentHandlers as $index => $handler) {
                if ($handler[0] === __CLASS__ && $handler[1] === 'parseContent') {
                    $currentIndex = $index;
                    break;
                }
            }
            
            // 如果当前插件是最后一个处理器，需要处理Markdown解析
            if ($currentIndex === count($contentHandlers) - 1) {
                $content = $widget->isMarkdown ? $widget->markdown($content) : $widget->autoP($content);
            }
        }
        
        return $content;
    }

    /**
     * 处理文章摘要
     *
     * @access public
     * @param string $excerpt 文章摘要
     * @param Widget_Abstract_Contents $widget 内容对象
     * @return string 处理后的摘要
     */
    public static function parseExcerpt($excerpt, $widget)
    {
        return self::parseShortCode($excerpt);
    }
    
    /**
     * 输出插件CSS样式和JavaScript代码
     *
     * @access public
     * @return string CSS样式和JavaScript代码
     */
    public static function outputHeader()
    {
        // 获取插件目录URL
        $pluginUrl = Helper::options()->pluginUrl;
        $cssUrl = $pluginUrl . '/ShortCodeIframe/style.css';
        $jsUrl = $pluginUrl . '/ShortCodeIframe/script.js';
        
        // 输出CSS链接
        echo '<link rel="stylesheet" href="' . $cssUrl . '" type="text/css" />';
        
        // 输出JavaScript链接
        // Artplayer相关库
        echo '<script type="text/javascript" src="https://registry.npmmirror.com/artplayer/latest/files/dist/artplayer.js"></script>';
        echo '<script type="text/javascript" src="https://registry.npmmirror.com/hls.js/latest/files/dist/hls.min.js"></script>';
        echo '<script type="text/javascript" src="https://registry.npmmirror.com/flv.js/latest/files/dist/flv.min.js"></script>';
        echo '<script type="text/javascript" src="https://registry.npmmirror.com/artplayer-plugin-danmuku/latest/files/dist/artplayer-plugin-danmuku.js"></script>';
        // 插件主JS
        echo '<script type="text/javascript" src="' . $jsUrl . '"></script>';
    }
}